<?php
if (!defined('ABSPATH')) exit;

class GPLCenter_Updater {
    private $cache_key = 'gplcenter_updater_cache';
    private $plugin_slug = 'gpl-center-updater/gpl-center-updater.php';
    
    public function __construct() {
        add_filter('pre_set_site_transient_update_plugins', [$this, 'check_for_updates'], 10, 1);
        add_action('plugin_row_meta', [$this, 'add_manual_check_link'], 10, 2);
        add_action('admin_init', [$this, 'handle_manual_check']);
    }

    public function check_for_updates($transient) {
        if (empty($transient->checked)) {
            return $transient;
        }

        $plugin_data = get_plugin_data(WP_PLUGIN_DIR . '/' . $this->plugin_slug);
        $current_version = $plugin_data['Version'];

        $remote_json = get_transient($this->cache_key);
        
        if (false === $remote_json) {
            $response = wp_remote_get('https://gplcenter.com/updater/GplCenter.json', [
                'timeout' => 15,
                'sslverify' => true,
                'headers' => ['Accept' => 'application/json']
            ]);

            if (!is_wp_error($response) && 
                wp_remote_retrieve_response_code($response) === 200 && 
                !empty($response['body'])) {
                
                $remote_json = json_decode($response['body'], true);
                
                if (isset($remote_json['latest_version'], $remote_json['download_url'])) {
                    set_transient($this->cache_key, $remote_json, 12 * HOUR_IN_SECONDS);
                }
            }
        }

        if (is_array($remote_json) && version_compare($current_version, $remote_json['latest_version'], '<')) {
            $transient->response[$this->plugin_slug] = (object) [
                'slug' => 'gpl-center-updater',
                'plugin' => $this->plugin_slug,
                'new_version' => $remote_json['latest_version'],
                'package' => $remote_json['download_url'],
                'tested' => $remote_json['tested'] ?? '',
                'requires' => $remote_json['requires'] ?? ''
            ];
        }

        return $transient;
    }

    public function handle_manual_check() {
        if (isset($_GET['force_check_gplcenter_updates']) && current_user_can('update_plugins')) {
            delete_transient($this->cache_key);
            wp_clean_plugins_cache();
            set_site_transient('update_plugins', null);
            
            wp_redirect(admin_url('plugins.php'));
            exit;
        }
    }

    public function add_manual_check_link($links, $file) {
        if ($file === $this->plugin_slug && current_user_can('update_plugins')) {
            $check_url = add_query_arg('force_check_gplcenter_updates', '1', admin_url('plugins.php'));
            
            $links[] = sprintf(
                '<a href="%s" title="%s"><span class="dashicons dashicons-update"></span> %s</a>',
                esc_url($check_url),
                esc_attr__('Manually check for new updates', 'gplcenter'),
                esc_html__('Check for updates', 'gplcenter')
            );
        }
        return $links;
    }
}

new GPLCenter_Updater();