<?php
/*
Plugin Name: GPL Center Updater
Plugin URI: https://www.gplcenter.com/
Description: GPL Center Auto Updater
Version: 2.6
Author: GPL Center
Author URI: https://www.gplcenter.com/
License: GPLv2 or later
Requires at least: 4.7
Requires PHP: 7.0
*/

// Prevent direct access to the file
if (!defined('ABSPATH')) {
    exit;
}

// Include the second plugin file
require_once plugin_dir_path(__FILE__) . 'custom-updates-manager.php';

function enqueue_subscription_styles() {
    wp_enqueue_style('subscription-style', plugin_dir_url(__FILE__) . 'assets/style.css');
}
add_action('admin_enqueue_scripts', 'enqueue_subscription_styles');

function enqueue_subscription_scripts() {
    wp_enqueue_script('jquery');
    wp_enqueue_script('subscription-notifications', plugin_dir_url(__FILE__) . 'assets/notifi.js', array('jquery'), null, true);
}
add_action('admin_enqueue_scripts', 'enqueue_subscription_scripts');

// Add settings menu in the dashboard
add_action('admin_menu', 'check_subscription_status_menu');

function check_subscription_status_menu() {
    add_menu_page(
        'GPL Center Authentication',
        'GPL Center',
        'manage_options',
        'check-subscription-status',
        'check_subscription_status_page',
        'dashicons-admin-plugins',
        6
    );

    add_submenu_page(
        'check-subscription-status',
        'Settings',
        'Settings',
        'manage_options',
        'cum-settings',
        'cum_render_settings_page'
    );
}

// Check and create table
add_action('plugins_loaded', 'check_and_create_table');

function check_and_create_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'gpl_center_options';

    if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
        create_subscription_status_table();
    }
}

function create_subscription_status_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'gpl_center_options';
    $charset_collate = $wpdb->get_charset_collate();

    $sql = "CREATE TABLE $table_name (
        option_id bigint(20) NOT NULL AUTO_INCREMENT,
        option_name varchar(191) NOT NULL,
        option_value longtext NOT NULL,
        autoload varchar(20) NOT NULL DEFAULT 'yes',
        PRIMARY KEY (option_id),
        UNIQUE KEY option_name (option_name)
    ) $charset_collate;";

    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    dbDelta($sql);
}

function add_default_options() {
    $default_options = array(
        'status' => '',
        'next_payment_date' => '',
        'product_name' => '',
        'token' => '',
        'ids' => '',
        'pt_list' => '',
        'sub' => '',
    );

    foreach ($default_options as $option_name => $option_value) {
        $existing_option = get_gpl_server_option($option_name, false);
        if ($existing_option === false) {
            update_gpl_server_option($option_name, $option_value);
        }
    }
}

function upgrade_gpl_server_table() {
    global $wpdb;
    $table_name = $wpdb->prefix . 'gpl_center_options';

    if ($wpdb->get_var("SHOW TABLES LIKE '$table_name'") != $table_name) {
        create_subscription_status_table();
    }

    $existing_data = $wpdb->get_var("SELECT COUNT(*) FROM $table_name");
    if ($existing_data == 0) {
        add_default_options();
    }
}
register_activation_hook(__FILE__, 'upgrade_gpl_server_table');

function update_gpl_server_option($option_name, $option_value, $autoload = 'yes') {
    global $wpdb;
    $table_name = $wpdb->prefix . 'gpl_center_options';

    $existing_option = $wpdb->get_row($wpdb->prepare("SELECT * FROM $table_name WHERE option_name = %s", $option_name));

    if ($existing_option) {
        $wpdb->update(
            $table_name,
            array(
                'option_value' => maybe_serialize($option_value),
                'autoload' => $autoload,
            ),
            array('option_name' => $option_name)
        );
    } else {
        $wpdb->insert(
            $table_name,
            array(
                'option_name' => $option_name,
                'option_value' => maybe_serialize($option_value),
                'autoload' => $autoload,
            )
        );
    }

    if ($wpdb->last_error) {
        error_log('Database error: ' . $wpdb->last_error);
    }
}

function get_gpl_server_option($option_name, $default = false) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'gpl_center_options';

    $option_value = $wpdb->get_var($wpdb->prepare("SELECT option_value FROM $table_name WHERE option_name = %s", $option_name));

    if ($option_value === null) {
        return $default;
    }

    $unserialized = maybe_unserialize($option_value);
    return $unserialized !== false ? $unserialized : $option_value;
}

function delete_gpl_server_option($option_name) {
    global $wpdb;
    $table_name = $wpdb->prefix . 'gpl_center_options';

    $wpdb->delete(
        $table_name,
        array('option_name' => $option_name)
    );

    if ($wpdb->last_error) {
        error_log('Database error: ' . $wpdb->last_error);
    }
}

// Settings page
function check_subscription_status_page() {
    global $wpdb;

    $email = isset($_POST['email']) ? sanitize_email($_POST['email']) : '';
    $password = isset($_POST['password']) ? sanitize_text_field($_POST['password']) : '';

    $status = get_gpl_server_option('status', false);
    $is_subscription_exists = !empty($status);
    $readonly = $is_subscription_exists ? 'readonly' : '';
    $deactivate_disabled = !$is_subscription_exists ? 'disabled' : '';
    $activate_disabled = $is_subscription_exists ? 'disabled' : '';

    if ($is_subscription_exists) {
        $email = '*********';
        $password = '*********';
    }

    if (isset($_POST['check_status'])) {
        $subscription_status = check_subscription_status($email, $password);
        echo '<div class="notification-message" data-type="success">' . esc_html($subscription_status) . '</div>';

        if ($subscription_status === 'Subscription activated successfully.') {
            echo '<script>
                    setTimeout(function() {
                        window.location.reload();
                    }, 3000);
                  </script>';
        }
    }

    if (isset($_POST['deactivate'])) {
        $deactivation_result = deactivate_subscription($email);
        echo '<div class="notification-message" data-type="info">' . esc_html($deactivation_result) . '</div>';

        if ($deactivation_result === 'Deactivation successful') {
            echo '<script>
                    setTimeout(function() {
                        window.location.reload();
                    }, 3000);
                  </script>';
        }
    }

    ?>
    <div class="wrap">
        <div class="subscription-box">
            <div class="gplt-login-header">
                <div class="gplt-brand">
                    <div class="gplt-logo-wrapper">
                        <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="40" height="40" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round" class="gplt-logo">
                            <path d="M20 7h-7c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2h7c1.1 0 2-.9 2-2V9c0-1.1-.9-2-2-2z" fill="#BC5994" stroke="#BC5994"></path>
                            <path d="M14 7V5c0-1.1-.9-2-2-2H5c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2h2" stroke="#393E46"></path>
                            <circle cx="16.5" cy="15.5" r="2.5" fill="white"></circle>
                        </svg>
                    </div>
                    <div class="gplt-brand-text">
                        <h1>GPL Center Authentication</h1>
                        <p>Activate your GPL Updater</p>
                    </div>
                </div>
            </div>
            <div class="subscription-form">
                <div class="gplt-login-info">Enter your GPL Server Email and password</div>
                <div class="subscription-form-s">
                    <form method="post" action="">
                        <td><div class="gplt-input-group"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="20" height="20" class="gplt-input-icon"><path fill="#BC5994" d="M12 12c2.21 0 4-1.79 4-4s-1.79-4-4-4-4 1.79-4 4 1.79 4 4 4zm0 2c-2.67 0-8 1.34-8 4v2h16v-2c0-2.66-5.33-4-8-4z"></path></svg>
                        <input type="email" class="gplt-input" id="email" name="email" value="<?php echo esc_attr($email); ?>" required <?php echo $readonly; ?> placeholder="Email"></div></td>
                        <td><div class="gplt-input-group"><svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="20" height="20" class="gplt-input-icon"><path fill="#BC5994" d="M18 8h-1V6c0-2.76-2.24-5-5-5S7 3.24 7 6v2H6c-1.1 0-2 .9-2 2v10c0 1.1.9 2 2 2h12c1.1 0 2-.9 2-2V10c0-1.1-.9-2-2-2zm-6 9c-1.1 0-2-.9-2-2s.9-2 2-2 2 .9 2 2-.9 2-2 2zm3.1-9H8.9V6c0-1.71 1.39-3.1 3.1-3.1 1.71 0 3.1 1.39 3.1 3.1v2z"></path></svg>
                        <input type="password" class="gplt-input" name="password" id="password" value="<?php echo esc_attr($password); ?>" required <?php echo $readonly; ?> placeholder="Password"></div></td>
                        <input type="submit" name="check_status" value="Activate" <?php echo $activate_disabled; ?>>
                        <input type="submit" name="deactivate" value="Deactivate" <?php echo $deactivate_disabled; ?> onclick="return confirm('Are you sure you want to deactivate?');">
                    </form>
                </div>
            </div>
            <div class="gplt-info-card <?php echo ($is_subscription_exists && $status === 'active') ? 'active' : 'inactive'; ?>">
                <div class="gplt-info-icon <?php echo ($is_subscription_exists && $status === 'active') ? 'active' : 'inactive'; ?>">
                    <?php
                    if ($is_subscription_exists) {
                        echo '<svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="18" height="18" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <path d="M22 11.08V12a10 10 0 1 1-5.93-9.14"></path>
                                    <polyline points="22 4 12 14.01 9 11.01"></polyline>
                                </svg>';
                    } else {
                        echo ' <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="18" height="18" fill="none" stroke="white" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                                    <circle cx="12" cy="12" r="10"></circle>
                                    <line x1="12" y1="16" x2="12" y2="12"></line>
                                    <line x1="12" y1="8" x2="12.01" y2="8"></line>
                                </svg>';
                    }
                    ?>
                </div>
                <div class="gplt-info-text">
                    <?php
                    if ($is_subscription_exists) {
                        echo '<span>Status: <strong> ' . esc_html($status) . '</strong></span>';
                        $next_payment_date = get_gpl_server_option('next_payment_date', false);
                        if ($next_payment_date) {
                            $next_payment_date = date('Y-m-d', strtotime($next_payment_date));
                            echo '<p><span>Expiration date: <strong> '  . esc_html($next_payment_date) . '</strong></span> </p>';
                        }
                        $product_name = get_gpl_server_option('product_name', false);
                        if ($product_name) {
                            echo '<p><span>Subscription type: <strong> '  . esc_html($product_name) .  '</strong></span> </p>';
                        }
                    } else {
                        echo '  <span>Status: <strong>Not Activated</strong></span>';
                    }
                    ?>
                </div>
            </div>
        </div>
    </div>
    <?php
}

function deactivate_subscription($email) {
    update_gpl_server_option('status', '');
    return 'Deactivation successful';
}

// Function to check subscription status (manual with email/password)
function check_subscription_status($email, $password) {
    $api_url = 'https://gplcenter.com/gpl_subscription.php';

    $args = array(
        'body' => array(
            'email' => $email,
            'password' => $password,
        ),
        'timeout' => 15,
    );

    $response = wp_remote_post($api_url, $args);

    if (is_wp_error($response)) {
        return 'Server connection error: ' . $response->get_error_message();
    }

    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    if (isset($data['subscriptions']) && is_array($data['subscriptions'])) {
        $active_subscriptions = array_filter($data['subscriptions'], function($subscription) {
            return $subscription['status'] === 'active';
        });

        if (!empty($active_subscriptions)) {
            $premium_subscriptions = array();
            $regular_subscriptions = array();
            $pt_list = array(
                'plugins' => array(),
                'themes' => array()
            );

            // Process download_logs
            if (isset($data['download_logs']) && is_array($data['download_logs'])) {
                if (!function_exists('get_plugins')) {
                    require_once ABSPATH . 'wp-admin/includes/plugin.php';
                }
                $all_plugins = get_plugins();

                foreach ($data['download_logs'] as $log) {
                    if (is_array($log) && count($log) >= 2) {
                        $name = $log[0];
                        $type = $log[1];
                        
                        if ($type === 'themes') {
                            $pt_list['themes'][] = sanitize_title($name);
                        } 
                        elseif ($type === 'plugins') {
                            $plugin_file = find_plugin_main_file($name, $all_plugins);
                            if ($plugin_file) {
                                $pt_list['plugins'][] = $plugin_file;
                            }
                        }
                    }
                }
            }

            // Classify subscriptions
            foreach ($active_subscriptions as $subscription) {
                $is_premium = isset($subscription['slug_update']) && 
                             in_array($subscription['slug_update'], array('m1', 'm6', 'y1'));

                if ($is_premium) {
                    $premium_subscriptions[] = $subscription;
                } else {
                    $regular_subscriptions[] = $subscription;
                    
                    if (isset($subscription['slug_update'])) {
                        if (strpos($subscription['slug_update'], '-theme') !== false) {
                            $pt_list['themes'][] = str_replace('-theme', '', $subscription['slug_update']);
                        } else {
                            $plugin_file = find_plugin_main_file($subscription['slug_update'], $all_plugins);
                            if ($plugin_file) {
                                $pt_list['plugins'][] = $plugin_file;
                            }
                        }
                    }
                }
            }

            $selected_subscription = null;
            $sub_value = 0;

            if (!empty($premium_subscriptions)) {
                usort($premium_subscriptions, function($a, $b) {
                    return strtotime($b['next_payment_date']) - strtotime($a['next_payment_date']);
                });
                $selected_subscription = $premium_subscriptions[0];
                $sub_value = 1;
            } 
            elseif (!empty($regular_subscriptions)) {
                usort($regular_subscriptions, function($a, $b) {
                    return strtotime($b['next_payment_date']) - strtotime($a['next_payment_date']);
                });
                $selected_subscription = $regular_subscriptions[0];
                $sub_value = 0;
                
                foreach ($regular_subscriptions as $subscription) {
                    if (isset($subscription['slug_update'])) {
                        if (strpos($subscription['slug_update'], '-theme') !== false) {
                            $pt_list['themes'][] = str_replace('-theme', '', $subscription['slug_update']);
                        } else {
                            $plugin_file = find_plugin_main_file($subscription['slug_update'], $all_plugins);
                            if ($plugin_file) {
                                $pt_list['plugins'][] = $plugin_file;
                            }
                        }
                    }
                }
            }

            if ($selected_subscription) {
                $pt_list['plugins'] = array_unique($pt_list['plugins']);
                $pt_list['themes'] = array_unique($pt_list['themes']);
                
                save_or_update_subscription_data(
                    $selected_subscription['status'],
                    $selected_subscription['next_payment_date'] ?? '',
                    $selected_subscription['product_name'] ?? '',
                    $data['token'] ?? '',
                    $selected_subscription['id'] ?? '',
                    $pt_list,
                    $sub_value
                );

                return 'Subscription activated successfully.';
            }
        }
        return 'No active subscriptions found.';
    }
    return 'Error: ' . $body;
}

// Function to check subscription status with token (for cron job)
function check_subscription_status_with_token($token) {
    $api_url = 'https://gplcenter.com/gpl_subscriptionxx.php';

    $args = array(
        'body' => array(
            'token' => $token,
        ),
        'timeout' => 15,
    );

    $response = wp_remote_post($api_url, $args);

    if (is_wp_error($response)) {
        error_log('Error in token subscription check: ' . $response->get_error_message());
        return 'Server connection error: ' . $response->get_error_message();
    }

    $body = wp_remote_retrieve_body($response);
    $data = json_decode($body, true);

    if (isset($data['subscriptions']) && is_array($data['subscriptions'])) {
        $active_subscriptions = array_filter($data['subscriptions'], function($subscription) {
            return $subscription['status'] === 'active';
        });

        if (!empty($active_subscriptions)) {
            $premium_subscriptions = array();
            $regular_subscriptions = array();
            $pt_list = array(
                'plugins' => array(),
                'themes' => array()
            );

            // Process download_logs if available
            if (isset($data['download_logs']) && is_array($data['download_logs'])) {
                if (!function_exists('get_plugins')) {
                    require_once ABSPATH . 'wp-admin/includes/plugin.php';
                }
                $all_plugins = get_plugins();

                foreach ($data['download_logs'] as $log) {
                    if (is_array($log) && count($log) >= 2) {
                        $name = $log[0];
                        $type = $log[1];
                        
                        if ($type === 'themes') {
                            $pt_list['themes'][] = sanitize_title($name);
                        } 
                        elseif ($type === 'plugins') {
                            $plugin_file = find_plugin_main_file($name, $all_plugins);
                            if ($plugin_file) {
                                $pt_list['plugins'][] = $plugin_file;
                            }
                        }
                    }
                }
            }

            // Classify subscriptions
            foreach ($active_subscriptions as $subscription) {
                $is_premium = isset($subscription['slug_update']) && 
                             in_array($subscription['slug_update'], array('m1', 'm6', 'y1'));

                if ($is_premium) {
                    $premium_subscriptions[] = $subscription;
                } else {
                    $regular_subscriptions[] = $subscription;
                    
                    if (isset($subscription['slug_update'])) {
                        if (strpos($subscription['slug_update'], '-theme') !== false) {
                            $pt_list['themes'][] = str_replace('-theme', '', $subscription['slug_update']);
                        } else {
                            $plugin_file = find_plugin_main_file($subscription['slug_update'], $all_plugins);
                            if ($plugin_file) {
                                $pt_list['plugins'][] = $plugin_file;
                            }
                        }
                    }
                }
            }

            $selected_subscription = null;
            $sub_value = 0;

            if (!empty($premium_subscriptions)) {
                usort($premium_subscriptions, function($a, $b) {
                    return strtotime($b['next_payment_date']) - strtotime($a['next_payment_date']);
                });
                $selected_subscription = $premium_subscriptions[0];
                $sub_value = 1;
            } 
            elseif (!empty($regular_subscriptions)) {
                usort($regular_subscriptions, function($a, $b) {
                    return strtotime($b['next_payment_date']) - strtotime($a['next_payment_date']);
                });
                $selected_subscription = $regular_subscriptions[0];
                $sub_value = 0;
                
                foreach ($regular_subscriptions as $subscription) {
                    if (isset($subscription['slug_update'])) {
                        if (strpos($subscription['slug_update'], '-theme') !== false) {
                            $pt_list['themes'][] = str_replace('-theme', '', $subscription['slug_update']);
                        } else {
                            $plugin_file = find_plugin_main_file($subscription['slug_update'], $all_plugins);
                            if ($plugin_file) {
                                $pt_list['plugins'][] = $plugin_file;
                            }
                        }
                    }
                }
            }

            if ($selected_subscription) {
                $pt_list['plugins'] = array_unique($pt_list['plugins']);
                $pt_list['themes'] = array_unique($pt_list['themes']);
                
                return array(
                    'status' => $selected_subscription['status'],
                    'next_payment_date' => $selected_subscription['next_payment_date'] ?? '',
                    'product_name' => $selected_subscription['product_name'] ?? '',
                    'id' => $selected_subscription['id'] ?? '',
                    'pt_list' => $pt_list,
                    'sub' => $sub_value
                );
            }
        }
        return 'No active subscriptions found.';
    }
    return 'Error: ' . $body;
}

// تعديل دالة الكرون لتستخدم الآلية الجديدة
function check_subscription_status_cron() {
    global $wpdb;

    // جلب التوكين من قاعدة البيانات
    $token = get_gpl_server_option('token', '');
    
    if (empty($token)) {
        error_log('No token found in database for cron check.');
        return;
    }

    // التحقق من حالة الاشتراكات باستخدام التوكين
    $subscription_data = check_subscription_status_with_token($token);

    if (is_array($subscription_data)) {
        // حفظ البيانات المحدثة
        save_or_update_subscription_data(
            $subscription_data['status'],
            $subscription_data['next_payment_date'],
            $subscription_data['product_name'],
            $token,
            $subscription_data['id'],
            $subscription_data['pt_list'],
            $subscription_data['sub']
        );

        error_log('Cron job: Subscription status updated successfully.');
    } else {
        error_log('Cron job error: ' . $subscription_data);
    }
}

// Helper function to find plugin main file
function find_plugin_main_file($plugin_slug, $all_plugins) {
    $direct_path = $plugin_slug . '/' . $plugin_slug . '.php';
    if (isset($all_plugins[$direct_path])) {
        return $direct_path;
    }

    foreach ($all_plugins as $plugin_file => $plugin_data) {
        $folder_name = dirname($plugin_file);
        if ($folder_name === $plugin_slug) {
            return $plugin_file;
        }
    }

    foreach ($all_plugins as $plugin_file => $plugin_data) {
        if (isset($plugin_data['TextDomain']) && $plugin_data['TextDomain'] === $plugin_slug) {
            return $plugin_file;
        }
        
        if (isset($plugin_data['Name'])) {
            $plugin_name_slug = sanitize_title($plugin_data['Name']);
            if ($plugin_name_slug === $plugin_slug) {
                return $plugin_file;
            }
        }
    }

    return $plugin_slug . '/' . $plugin_slug . '.php';
}

function save_or_update_subscription_data($status, $next_payment_date, $product_name, $token = '', $ids = '', $pt_list = '', $sub = 0) {
    if (is_string($pt_list) && !empty($pt_list)) {
        $unserialized = maybe_unserialize($pt_list);
        if (is_array($unserialized)) {
            $pt_list = $unserialized;
        }
    }
    
    $pt_list = is_array($pt_list) ? $pt_list : array();
    
    update_gpl_server_option('status', $status);
    update_gpl_server_option('next_payment_date', $next_payment_date);
    update_gpl_server_option('product_name', $product_name);
    update_gpl_server_option('token', $token);
    update_gpl_server_option('ids', $ids);
    update_gpl_server_option('pt_list', $pt_list);
    update_gpl_server_option('sub', $sub);
}

function display_subscription_notices() {
    $current_screen = get_current_screen();

    if ($current_screen->id === 'toplevel_page_check-subscription-status') {
        return;
    }

    $status = get_gpl_server_option('status', false);

    if (empty($status)) {
        echo '
        <div class="notice notice-error is-dismissible custom-notice">
            <div class="notice-icon">
                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <circle cx="12" cy="12" r="10"></circle>
                    <line x1="12" y1="8" x2="12" y2="12"></line>
                    <line x1="12" y1="16" x2="12.01" y2="16"></line>
                </svg>
            </div>
            <div class="notice-content">
                <h3>Plugin Activation Required!</h3>
                <p>You must activate the plugin to enable automatic updates. <a href="' . admin_url('admin.php?page=check-subscription-status') . '">Activate Now</a></p>
            </div>
        </div>';
    } elseif ($status !== 'active') {
        echo '
        <div class="notice notice-warning is-dismissible custom-notice">
            <div class="notice-icon">
                <svg xmlns="http://www.w3.org/2000/svg" viewBox="0 0 24 24" width="24" height="24" fill="none" stroke="currentColor" stroke-width="2" stroke-linecap="round" stroke-linejoin="round">
                    <path d="M10.29 3.86L1.82 18a2 2 0 0 0 1.71 3h16.94a2 2 0 0 0 1.71-3L13.71 3.86a2 2 0 0 0-3.42 0z"></path>
                    <line x1="12" y1="9" x2="12" y2="13"></line>
                    <line x1="12" y1="17" x2="12.01" y2="17"></line>
                </svg>
            </div>
            <div class="notice-content">
                <h3>Subscription Expired!</h3>
                <p>Please renew your subscription to receive updates. <a href="' . admin_url('admin.php?page=check-subscription-status') . '">Renew Now</a></p>
            </div>
        </div>';
    }
}
add_action('admin_notices', 'display_subscription_notices');

// Register cron events
register_activation_hook(__FILE__, 'gpl_schedule_subscription_cron');
register_deactivation_hook(__FILE__, 'gpl_deactivate_subscription_cron');

function gpl_schedule_subscription_cron() {
    if (!wp_next_scheduled('gplcenter_subscription_check')) {
        wp_schedule_event(time(), 'daily', 'gplcenter_subscription_check');
        error_log('GPL Center Updater: Cron event registered successfully.');
    }
}

function gpl_deactivate_subscription_cron() {
    wp_clear_scheduled_hook('gplcenter_subscription_check');
    error_log('GPL Center Updater: Cron event deactivated successfully.');
}

// Hook the cron function
add_action('gplcenter_subscription_check', 'check_subscription_status_cron');

require_once plugin_dir_path(__FILE__) . 'includes/class-updater.php';